<?php
/**
 * This file specifically checks which user role has access to certain Elementor controls.
 *
 * @link       https://posimyth.com/
 * @since      2.5.1
 *
 * @package    Uichemy
 */

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Plugin;

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Uich_Elementor' ) ) {

	/**
	 * Main class for implementing custom user role functionality in Elementor.
	 *
	 * @link       https://posimyth.com/
	 * @since      2.5.1
	 */
	class Uich_Elementor {

		/**
		 * Initialize the core functionality of the plugin.
		 * Set up necessary hooks and actions.
		 *
		 * @since 2.5.1
		 */
		public function __construct() {

			if ( defined( 'ELEMENTOR_VERSION' ) && empty( get_option('uich_elementor_custom_css') ) ) {
				add_action( 'elementor/init', array( $this, 'uich_initialize_controls' ) );
			}
		}

		/**
		 * Initialize Elementor controls and register necessary actions.
		 *
		 * @since 2.5.1
		 */
		public function uich_initialize_controls() {
			// Add custom CSS controls to specific sections.
			add_action( 'elementor/element/common/_section_responsive/after_section_end', array( $this, 'uich_add_custom_css_controls' ), 10, 2 );
			add_action( 'elementor/element/section/_section_responsive/after_section_end', array( $this, 'uich_add_custom_css_controls' ), 10, 2 );
			add_action( 'elementor/element/column/_section_responsive/after_section_end', array( $this, 'uich_add_custom_css_controls' ), 10, 2 );
			add_action( 'elementor/element/container/_section_responsive/after_section_end', array( $this, 'uich_add_custom_css_controls' ), 10, 2 );

			add_action( 'elementor/element/parse_css', array( $this, 'uich_apply_post_css' ), 10, 2 );

			add_action( 'elementor/editor/after_enqueue_scripts', array( $this, 'uich_enqueue_panel_scripts' ), 988 );
		}

		/**
		 * Enqueue scripts for the Elementor editor panel.
		 *
		 * This function enqueues the custom JavaScript file used in the Elementor editor panel,
		 * ensuring dependencies like jQuery and wp-i18n are loaded. The script is added with the
		 * correct version and loaded in the footer for optimized performance.
		 *
		 * @since 2.5.1
		 */
		public function uich_enqueue_panel_scripts() {
			wp_enqueue_script(
				'uich-addons-editor-js',
				UICH_URL . 'assets/js/uich-elementor-editor.js',
				array( 'jquery', 'wp-i18n' ),
				UICH_VERSION,
				true
			);
		}

		/**
		 * Adds custom CSS controls to Elementor widgets.
		 * This will be shown in the advanced tab of the Elementor editor.
		 *
		 * @since 2.5.1
		 *
		 * @param \Elementor\Widget_Base $element The Elementor widget element.
		 */
		public function uich_add_custom_css_controls( $element ) {

			if ( ! current_user_can( 'edit_pages' ) && ! current_user_can( 'unfiltered_html' ) ) {
				return;
			}

			$element->start_controls_section(
				'custom_css_section',
				array(
					'label' => esc_html__( 'UiChemy : Custom CSS', 'uichemy' ),
					'tab'   => Controls_Manager::TAB_ADVANCED,
				)
			);
			$element->add_control(
				'uich_custom_css_field',
				array(
					'label'       => esc_html__( 'CSS Code', 'uichemy' ),
					'type'        => Controls_Manager::CODE,
					'language'    => 'css',
					'render_type' => 'ui',
					'separator'   => 'none',
					'rows'        => 5,
				)
			);
			$element->end_controls_section();
		}

		/**
		 * Applies the custom CSS generated by the user in the widget to the front-end.
		 *
		 * @since 2.5.1
		 *
		 * @param \Elementor\Core\DynamicTags\Dynamic_CSS $css The CSS object.
		 * @param \Elementor\Widget_Base                  $element The widget element being rendered.
		 */
		public function uich_apply_post_css( $css, $element ) {

			if ( $css instanceof \Elementor\Core\DynamicTags\Dynamic_CSS ) {
				return;
			}

			$element_settings = $element->get_settings();

			$processed_css = $this->uich_generate_css( $element_settings, $css->get_element_unique_selector( $element ) );

			$css->get_stylesheet()->add_raw_css( $processed_css );
		}

		/**
		 * Generates the custom CSS based on the settings and unique selector.
		 *
		 * @since 2.5.1
		 *
		 * @param array  $element_settings The widget's settings.
		 * @param string $unique_selector The unique selector for the widget.
		 * @return string The processed custom CSS.
		 */
		public function uich_generate_css( $element_settings, $unique_selector ) {

			if ( empty( $element_settings['uich_custom_css_field'] ) ) {
				return;
			}

			$custom_css = trim( $element_settings['uich_custom_css_field'] );

			if ( empty( $custom_css ) ) {
				return;
			}

			$custom_css = str_replace( 'selector', $unique_selector, $custom_css );

			return wp_strip_all_tags( $custom_css );
		}
	}

	new Uich_Elementor();

}